globals  [ticks-per-sec patches-per-tick-per-kph patches-per-km world-width-km
  number-of-cars min-speed-limit min-distance red-car max-accel speed-limit-ok-count]
breed    [cars car]
cars-own [speed tailgate-secs]

to setup
  clear-all
  set ticks-per-sec 30
  set min-speed-limit 30
  set min-distance .001
  set max-accel  .001 * 3600 / ticks-per-sec                            ;;  accel per tick ~  1m per sec per sec

  let patch-length 5                                                    ;; just big enough for a 5m car
  set patches-per-km 1000 / patch-length
  set world-width-km  world-width * patch-length / 1000
  let km-per-tick-per-kph  1 / 3600 / ticks-per-sec
  set patches-per-tick-per-kph  km-per-tick-per-kph * 1000 / patch-length
  ask patches with [abs pycor < 2] [ set pcolor white ]                 ;; setup-road
  setup-cars
end

to setup-cars
  clear-turtles
  clear-all-plots
  set-default-shape cars "car"
  set number-of-cars round (world-width * traffic-density / 100)
  foreach sort n-of number-of-cars patches with [pycor = 0] [ ?1 -> ask ?1 [sprout-cars 1 [
    set color blue
    set heading 90
    set speed 0
    set tailgate-secs 1 + random-float 1                                ;; 1-2 secs behind the car ahead
  ]] ]
  set red-car one-of cars
  ask red-car [ set color red ]
  watch red-car
  reset-ticks
end

to go
  foreach n-values number-of-cars [ ?1 -> number-of-cars - 1 - ?1 ] [ ?1 -> ask car ?1 [follow-car ] ]   ;;  Reduce chance of shunting by always moving the car ahead first (for all except the last car)
  tick
end

to follow-car       ;; car following:  adjust speed by targeting own tailgate-secs
  let car-ahead car ((who + 1) mod number-of-cars)
  let distance-to-car-ahead ((([xcor] of car-ahead - xcor) mod world-width) - 1) / patches-per-km  ;; bumper-to-bumper distance in km
  if speed > 0 and distance-to-car-ahead < 0 [error word " Shunt !!!   distance-to-car-ahead = " distance-to-car-ahead]  ;; in case safe-headway and brake-factor have not been set to avoid shunts

  let headway-secs max (list 0 (distance-to-car-ahead - min-distance)) / (speed + .000000001) * 3600   ;; Secs to travel headway (- min-distance) at the current speed (+ .000000001 to avoid ZeroDivide)
  let accel-factor max list -1 min list 1 (headway-secs - tailgate-secs)        ;; +ve if headway-secs > tailgate-secs, else -ve (capped at -1 to +1 to avoid excessive acceleration)

  ;; reset accel-factor for hard braking, if closer than safe-headway
  let closing-speed speed - [speed] of car-ahead
  let could-shunt (distance-to-car-ahead - min-distance) < (safe-headway / 3600 * closing-speed)
  if could-shunt [set accel-factor 0 - brake-factor]                            ;; hard brake if could-shunt

  set speed max list 0 min list speed-limit (speed + accel-factor * max-accel ) ;; observe the speeed-limit, and no reversing
  fd speed * patches-per-tick-per-kph
end

to optimise-speed-limits
  setup
  set traffic-density traffic-density - 1        ;; compensate for incrementing before the first optimise-speed-limit
  while [speed-limit >= min-speed-limit] [
    set traffic-density traffic-density + 1      ;; repeat for traffic-density = 2,3,...
    optimise-speed-limit
  ]
end

to optimise-speed-limit
  set speed-limit-ok-count 0
  while [speed-limit-ok-count < 300] [
    if speed-limit < min-speed-limit [
      output-print word "Finished optimising speed limits down to min-speed-limit of " min-speed-limit
      stop
    ]
    decrement-speed-limit
  ]
  let cars-per-hr number-of-cars * speed-limit / world-width-km
  output-print (word "For traffic-density " substring (word traffic-density " ") 0 2  "  speed-limit = "  substring (word speed-limit " ") 0 3  "   cars-per-hr = " cars-per-hr)
end

to decrement-speed-limit
  setup-cars

  let ticks-per-traverse-at-speed-limit  3600 * ticks-per-sec * world-width-km / speed-limit

  repeat 3 * ticks-per-traverse-at-speed-limit [
    go
    if all? cars [speed = speed-limit] [
      set speed-limit-ok-count speed-limit-ok-count + 1
      stop
    ]
  ]

  set speed-limit speed-limit - 1
end
@#$#@#$#@
GRAPHICS-WINDOW
10
130
1418
237
-1
-1
14.0
1
10
1
1
1
0
1
0
1
0
99
-3
3
1
1
1
ticks
30.0

BUTTON
10
80
65
113
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
10
30
200
63
traffic-density
traffic-density
2
40
23.0
1
1
%
HORIZONTAL

PLOT
10
265
1420
425
Red Car speed
ticks
speed
0.0
10.0
0.0
1.0
true
false
"set-plot-y-range 0 speed-limit\nstop" "if not plotting? [stop]"
PENS
"red car" 1.0 0 -2674135 true "" "plot [speed] of red-car"

SLIDER
205
30
420
63
speed-limit
speed-limit
min-speed-limit
120
29.0
1
1
kph
HORIZONTAL

SWITCH
425
30
515
63
plotting?
plotting?
0
1
-1000

BUTTON
335
80
445
113
NIL
optimise-speed-limits
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
70
80
125
113
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
220
80
330
113
NIL
optimise-speed-limit
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

OUTPUT
10
429
1420
839
12

SLIDER
530
30
702
63
safe-headway
safe-headway
2
20
10.0
1
1
secs
HORIZONTAL

SLIDER
715
30
887
63
brake-factor
brake-factor
1
5
3.0
1
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

This models the flow of traffic on a single-carriageway. It is different from other models in that the "car following" rules are simple, realistic and could mirror actual driving practices. Acceleration/deceleration is smooth and within comfortably achievable bounds for the average car. Each car adjusts its `headway-secs` (i.e. the number of seconds it is behind the `car-ahead` when passing a fixed point) towards its desired `tailgate-secs` (i.e. the target `headway-secs` that that an individual driver prefers).  A realistically achievable hard braking is applied whenever there is a chance of shunting the `car-ahead`.

The road is effectively a continuous loop, as the world wraps. The model demonstrates how wave-like traffic congestion forms, even without any accidents or road works.

## HOW TO USE IT

Set `traffic-density` to the desired %occupancy of the total road space. 

Click `setup` to create randomly distributed stationary cars with heterogoneous `tailgate-secs` (between 1 and 2 seconds, in line with the general actual driving practices in European countries, despite the recommended "two second rule"). 

Click `go-forever` to drive the cars. 

Experiment with `safe-headway` (the minimum seconds it is safe to follow at) and `brake-factor` (the factor by which the normal deceleration limit is multiplied by in order to brake hard when avoiding the chance of shunting the car ahead) to see how the "Red Car speed" plot varies, and what are the threshold values for avoiding shunts (which raise an error mssage and stop the run).  The default values of these parameters are the minimum ones that have been experimentally found to avoid shunting in all circumstances.  

The car following rules have been kept simple enough to be intuitively approximate-able by humans. Cars accelerate/decelerate by a factor based on `headway-secs - tailgate-secs`. Thus a car accelerates when `headway-secs > tailgate-secs` and decelerates when `headway-secs < tailgate-secs`.  This factor is bounded by the range -1 to +1 and multiplied by `max-accel` to give the acceleration at each tick, where `max-accel` is a realistic limit for normal acceleration and deceleration.  The factor is reset to - `brake-factor` when there is a possibility that the car `could-shunt` the `car-ahead`.  This causes the car to brake hard to avoid the possible shunt. The `could-shunt` test is basically a simple comparison of `distance-to-car-ahead` verses `closing-speed * safe-headway`, something which a human driver could feasibly approximate in his head.

Optimum speed limits for the fastest constant flow of traffic at each `traffic-density`, can be determined as follows:

Click `optimise-speed-limit` to incrementally reduce the `speed-limit` from its current value until all cars can consistently reach this `speed-limit` after an initial settling down period.  Thereafter all cars  continue to travel at this constant speed and optimum throughput is achieved for the current density (the values of which are logged to the `output` area).

Click `optimise-speed-limits` to run `optimise-speed-limit` as the `traffic-density` is incrementally increased from its current value. So, as this proceeds the `traffic-density` increases and the `speed-limit` decreases.  It terminates when the `minimum-speed-limit` is reached. 

## THINGS TO NOTICE

Cars that are clustered together move slowly, causing cars behind them to slow down also. As this effect ripples back a traffic jam forms. Even though all of the cars are moving forward, the traffic jams often move backwards. This behavior is common in wave phenomena: the behavior of the group is often very different from the behavior of the individuals that make up the group.  An example random car is painted red and highlighted for easy watching. Its speed plot eventually settles down either to periodic oscillations, or to the speed limit.

## EXTENDING THE MODEL

Try other realistic car following rules in follow-car. 

## RELATED MODELS

This is an improvement of "Traffic Basic" which is patch-based, only looks one-patch ahead and decelerates impossibly jerkily in just one step to slower than the car-ahead.  In contrast this model looks as far ahead as the next car and accelerates / decelerates  smoothly and realistically.

## COPYRIGHT AND LICENSE
<a rel="license" href="http://creativecommons.org/licenses/by/4.0/"><img alt="Creative Commons Licence" style="border-width:0" src="https://i.creativecommons.org/l/by/4.0/88x31.png" /></a><br /><span xmlns:dct="http://purl.org/dc/terms/" property="dct:title"></span> by <span xmlns:cc="http://creativecommons.org/ns#" property="cc:attributionName">Ian J Heath</span> is licensed under a Creative Commons Attribution 4.0 International License
http://creativecommons.org/licenses/by/4.0/
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.2
@#$#@#$#@
setup
repeat 180 [ go ]
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="false">
    <go>experiment</go>
    <timeLimit steps="1"/>
    <metric>mean-cars-per-hr</metric>
    <steppedValueSet variable="speed-limit" first="50" step="10" last="120"/>
    <steppedValueSet variable="number-of-cars" first="20" step="2" last="40"/>
    <enumeratedValueSet variable="stdev-to-test-steady">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="plotting?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="stdev-to-test-settled-down">
      <value value="25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="moving-mean-width">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="#experiment-repititions">
      <value value="10"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
