globals [
  demand-today           ;; the demand occurs at all retailers everyday
  colors                 ;; colors for plot
  ]

breed [players player]

;; There are 2 types of directed links in the model
;; demand links for order placement, and supply links for shipment pipeline

directed-link-breed [supply-links supply-link]
supply-links-own [orders-filled pair-demand-link]

directed-link-breed [demand-links demand-link]
demand-links-own [orders-placed back-orders]

players-own [
  role             ;; 3 roles in the model: supplier, distributor, and retailer
  base-stock       ;; we assume players are operating under base-stock policy   
  on-hand
  backlog
  inventory-position   ;; inventory equals to on-hand inventory + pipeline inventory - back-orders
  last-received
  pen-color
  cost
  id-number         ;; id-number is used to distinguish players with the same role
  demand-history    ;; a list consisting of the demands in a few past days
  revenue           ;; revenue from selling the products
  safety-factor     ;; parameter determining to what extent the player want to keep safety inventory against demand uncertainty
  current-supplier  
]

to setup
  ca
  set colors [5 15 25 35 45 55 65 75 85 95 105 115 125 135 67 117 28 18 37]  ;; a list of 20 colors
  set colors shuffle colors   
  layout
  initialize
  resize-shape
  reset-plots
  reset-ticks
end

to go
  if ticks >= days-of-simulation [stop]
  set demand-today daily-demand          ;; generate today's demand
  place-order-to-up
  receive-order-from-up
  process-order-from-down
  summarize
  update-policy
  resize-shape
  tick
end

to update-policy             ;; the players can update their inventory policy according to their demand record
  ask players with [role = "distributor"][
    set base-stock cal-base-stock-level demand-history lead-time-supplier-distributor
  ]
  ask players with [role = "retailer"][
    set base-stock cal-base-stock-level demand-history lead-time-distributor-retailer
  ]
end

;; Inventory operations
;; **************************
to place-order-to-up
  ask players with [role = "retailer" or role = "distributor"][
                                                                       ;; in this model, we assume all players are operating under base-stock policy
    let amount-to-order max list (base-stock - inventory-position) 0   ;; that is once the inventory position is below the base stock level, 
                                                                       ;; we order the quantity equal to the difference between inventory position and base stock level
                                                                       ;; otherwise, don't place any order

    ask my-out-demand-links [set orders-placed 0]                      ;; since in a network supply chain, a player can order from multiple upstream nodes
    ask who-to-order [set orders-placed amount-to-order]               ;; choose one node each time
  ]
end

to receive-order-from-up 
  ask players [
    if role = "distributor" or role = "retailer" [                       
      set last-received sum [first orders-filled] of my-in-supply-links     ;; take out the first item in the supply-link pipeline
      ask my-in-supply-links [set orders-filled but-first orders-filled]    ;; remove it from the pipeline
      set on-hand on-hand + last-received                                   ;; add it to the current on-hand inventory
      
    ]
    if role = "supplier" [set on-hand 10000]                                ;; we assume the supplier has unlimited supply
  ]
end

to process-order-from-down
  ask players [
    let new-orders 0                                        ;; for distributors and suppliers, new orders equal to the sum of the orders-placed of all in-demand-links
    if role = "distributor" or role = "supplier" [set new-orders sum [orders-placed] of my-in-demand-links]   
    if role = "retailer" [set new-orders demand-today]      ;; for retailers, new orders simply equal to today's demand
 
    set demand-history lput new-orders demand-history       ;; record the new-order in the demand history    
    set demand-history but-first demand-history             ;; delete the earliest demand history on the record, in order to keep the record length the same
    
    let orders-requested new-orders + backlog               ;; besides new orders, back-orders also need to be satisfied
    let orders-to-ship min list orders-requested on-hand    ;; if there is sufficient inventory, ship the requested amount
                                                            ;; if not sufficient inventory, ship whatever on-hand
    
    if role = "distributor" [set revenue revenue + 2 * orders-to-ship]    ;; revenue for distributors is 2 for each unit shipped
    if role = "retailer" [set revenue revenue + 3 * orders-to-ship]       ;; revenue for is 3 for each unit shipped
    
    set backlog max list 0 (backlog - on-hand + new-orders)    ;; the unsatisfied demand is counted as backlog (or back-orders)
    
    let rest-amount orders-to-ship       ;; allocate total shipping amount to each downstream node
    
    foreach sort my-out-supply-links [
      ask ? [                                    ;; quota to each supply link is proportional the sum of back-orders and new orders of the pair demand link
        let quota sum [back-orders] of pair-demand-link + sum [orders-placed] of pair-demand-link   
        let ship-to-this-link 0                  ;; if no order, ship nothing, and put 0 in the supply link
        if orders-requested > 0 [                ;; if positive order, ship according to the quota
          set ship-to-this-link min list ceiling (quota * orders-to-ship / orders-requested) rest-amount
        ]                                        ;; note that we use ceiling to guarantee the integrity of the shipping quantity
        set rest-amount rest-amount - ship-to-this-link  
        set orders-filled lput ship-to-this-link orders-filled    ;; put the ship quantity at the last place of the supply pipeline
        ask pair-demand-link [set back-orders max list 0 (quota - ship-to-this-link)]  ;; update the back-orders in the pair demand link
      ]
    ]
    
    set on-hand on-hand - orders-to-ship    ;; reduce the shipped quantity from the on-hand inventory
  ]  
end




to summarize
  ask players [
    let pipeline sum [sum orders-filled] of my-in-supply-links    ;; calculate the pipeline inventory (inventory in-transit) for each player
    set inventory-position on-hand + pipeline - backlog           ;; recalculate the inventory position
    let cost-add (0.5 * on-hand + 2 * backlog)                    ;; calculate inventory holding cost and backlog penalty
    set cost cost + cost-add                                      ;; update the cost
  ]
  
  ask players with [role = "distributor" or role = "retailer"] [  ;; plots
    plot-on-hand-inventory
    plot-profit
  ]
end

;; ****************************




to-report y-locations [number]         ;; report a list of y-coordinates that would evenly distribute the "number" players on y-axis
  let y-list []

  let interval round ((2 * (max-pycor - 3) + 1) / (number + 1))
  
  let current-pos (- max-pycor + 3 + interval)
  
  repeat number [
    set y-list lput current-pos y-list
    set current-pos current-pos + interval
  ]
  report y-list
end


to layout
  set-default-shape players "circle"
  set-default-shape links "arc"

  create-players 1 [
    setxy -10 0
    set color red
    set role "supplier"
    set size 5                ;; the size of the supplier won't be updated in the simulation
  ] 
  
  ask patches with [pxcor = 0 and (member? pycor y-locations num-of-distributors)] [  ;; evenly layout the distributors in the middle
    sprout-players 1 [
      set color blue
      set role "distributor"      
    ]
  ]
  
  ask patches with [pxcor = 10 and (member? pycor y-locations num-of-retailers)] [ ;; evenly layout the retailers on the right
    sprout-players 1 [
      set color green
      set role "retailer"
    ]
  ]
  
  
  let d-number 1
  foreach sort players with [role = "distributor"][   ;; assign each distributor an id-number, and label them with the id-number
    ask ?[
      set label word "D-" d-number
      set id-number d-number
      set d-number d-number + 1
    ]
  ]
  
  let r-number 1
  foreach sort players with [role = "retailer"][    ;; assign each retailer an id-number, and label them with the id-number
    ask ?[
      set label word "R-" r-number
      set id-number r-number
      set r-number r-number + 1
    ]
  ]
  
  let index 0
  foreach sort players with [role = "distributor" or role = "retailer"][
    ask ?[
      set pen-color item index colors    ;; assign each distributor and retailer a different plot pen color
      set index index + 1
    ]
  ]
  
  
  
  ask players with [role = "retailer"] [
    create-demand-links-to players with [role = "distributor"]     ;; create demand and supply links between distributors and retailers
    create-supply-links-from players with [role = "distributor"]   
  ]
  ask players with [role = "distributor"] [
    create-demand-links-to players with [role = "supplier"]        ;; create demand and supply links between supplier and distributors
    create-supply-links-from players with [role = "supplier"] 
  ]
  
  ask supply-links [                                  ;; define the pair demand link of each supply link, the link with the same ends
    set pair-demand-link demand-links with [end1 = [end2] of myself and end2 = [end1] of myself]
  ]
  
  
  
end

to initialize
  
  ask players [
    if role = "distributor" [
      set base-stock initial-stock-distributor   ;; set the initial base-stock level and on-hand inventory to "initial-stock-distributor"
      set on-hand initial-stock-distributor
      ask my-out-demand-links [
        set orders-placed 0 
        set back-orders 0
        ]
      
      ;; create a list with all elements equal to 0, with the length equals to "lead-time-supplier-distributor"
      ;; representing the supply pipeline between the supplier and the distributor
      
      ask my-in-supply-links [set orders-filled n-values lead-time-supplier-distributor [0]]  
    ]
    if role = "retailer" [
      set base-stock initial-stock-retailer
      set on-hand initial-stock-retailer
      
      ask my-out-demand-links [
        set orders-placed 0
        set back-orders random 0
        ]
      
      ;; create a list with all elements equal to 0, with the length equals to "lead-time-distributor-retailer"
      ;; representing the supply pipeline between the distributor and the retailer
      
      ask my-in-supply-links [set orders-filled n-values lead-time-distributor-retailer [0]]
    ]
    if role = "supplier"[
      set on-hand 10000
    ]
    set backlog 0
    set inventory-position on-hand - backlog
  ]
  
  ask players [    ;; randomly assign the safety-factor to players
    set cost 0     ;; the higher safety factor  means that the player is willing to keep higher safety inventory against the demand uncertainty
    set revenue 0                                    
    set safety-factor 1.5 + random-float 1
    set demand-history n-values record-length [""]  ;; set the demand history as a list with empty elements, with the length equals to "record-length"
    ]

end

to resize-shape    ;; visualize the on-hand stock via size of the turtle
  ask players with [role = "distributor" or role = "retailer"][
    set size 0.5 * (sqrt on-hand)          
  ]
end


to reset-plots          ;; clear all plots and create plot pens for each player
  clear-all-plots
  ask players with [role = "distributor" or role = "retailer"][
    create-plot-pens
  ]
end

to create-plot-pen [my-plot]
  set-current-plot my-plot
  create-temporary-plot-pen label  ;; name the plot pen with the player's label
  set-plot-pen-color pen-color
end

to create-plot-pens
  create-plot-pen "total profit"       ;; one plot for total profit
  create-plot-pen "on-hand inventory"  ;; one plot for on-hand inventory level
end

to plot-profit
  set-current-plot "total profit"
  set-current-plot-pen label
  plot revenue - cost
end

to plot-on-hand-inventory
  set-current-plot "on-hand inventory"
  set-current-plot-pen label
  plot on-hand
end



;; Demand distribution generation
;; **************************************
to-report daily-demand                ;; we design three mechanisms for generating the daily demand, which can be chosen in the chooser
  if distribution = "deterministic"
    [report deterministic-demand]     ;; deterministic demand means the demand is constant, there is no uncertainty      
  
  if distribution = "poisson"         ;; poisson demand means that the daily demand follows Poisson distribution
    [report random-poisson mean-for-poisson]             
  
  if distribution = "normal"          ;; normal demand means that the daily demand follows truncated normal distribution (modified in this model)
    [report truncated-normal mean-for-normal std-for-normal lower-bound-for-normal upper-bound-for-normal]   
end


to-report truncated-normal [mean-value std-value min-value max-value]    ;; there are 4 parameters for the truncated normal distribution
  let random-num random-normal mean-value std-value                      ;; we first generate a random-normal number according to the mean value and standard-deviation value
  ifelse random-num > max-value or random-num < min-value                
  [report round (min-value + random-float (max-value - min-value))]      ;; if the value is beyond the min-value and max-value, report a random number within the range
  [report round random-num]                                              ;; if the value is within the min-value and max-value, report the rounding of this number
end
;; ***************************************************************





to-report cal-base-stock-level [demand-list delay]     ;; calculate base-stock based on demand-history
  let numbers filter is-number? demand-list            ;; during the first few days, not all the elements in the demand history are numbers, but ""
  let mean-value mean numbers                          ;; calculate mean value
  
  let std 0
  if length numbers >= 2[                              ;; calculate the standard deviation of the demand history
    set std standard-deviation numbers
  ]
                                                       ;; according to inventory theories, the base-stock level is usually calculate according to mean and std and supply delays                                  
  report round (mean-value * (delay + 1) + safety-factor * std * (sqrt (delay + 1)))  ;; "+1" because of the order processing delay
end


to-report who-to-order  ;; report the demand link that has the lowest back-orders, so that the player will choose to order from this one
  let min-back-order min [back-orders] of my-out-demand-links  ;; find out the minimum back-order
  let sorted-links []                         ;; prepare an empty list
  foreach sort my-out-demand-links [          ;; if the back-orders of the demand links equals to the minimum back-order
    ask ? [                                   ;; add the demand link to sorted-links list
      if back-orders = min-back-order [set sorted-links lput self sorted-links]
    ]
  ]
  ifelse member? current-supplier sorted-links [   ;; if the current-supplier (actually the corresponding demand link) is among the demand links with the minimum back-order 
    report current-supplier                        ;; choose the current supplier due to customer loyalty
  ][
  let chosen-one one-of sorted-links               ;; if not, choose one from the demand links with the minimum back-order
  set current-supplier chosen-one                  ;; then transfer the supplier to this one
  report chosen-one                               
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
550
451
16
20
10.0
1
10
1
1
1
0
0
0
1
-16
16
-20
20
1
1
1
ticks
30.0

BUTTON
18
14
82
48
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
114
16
178
50
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
112
58
188
92
go once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

CHOOSER
567
98
706
143
distribution
distribution
"deterministic" "poisson" "normal"
2

SLIDER
566
175
739
208
deterministic-demand
deterministic-demand
0
20
20
1
1
NIL
HORIZONTAL

SLIDER
566
244
739
277
mean-for-poisson
mean-for-poisson
0
20
15
1
1
NIL
HORIZONTAL

SLIDER
566
319
739
352
mean-for-normal
mean-for-normal
0
20
10
1
1
NIL
HORIZONTAL

SLIDER
567
357
740
390
std-for-normal
std-for-normal
0
10
1
1
1
NIL
HORIZONTAL

SLIDER
566
394
748
427
lower-bound-for-normal
lower-bound-for-normal
0
10
0
1
1
NIL
HORIZONTAL

SLIDER
566
432
751
465
upper-bound-for-normal
upper-bound-for-normal
0
100
100
1
1
NIL
HORIZONTAL

SLIDER
8
186
204
219
lead-time-supplier-distributor
lead-time-supplier-distributor
0
5
2
1
1
NIL
HORIZONTAL

SLIDER
2
232
208
265
lead-time-distributor-retailer
lead-time-distributor-retailer
0
5
2
1
1
NIL
HORIZONTAL

SLIDER
18
282
198
315
initial-stock-distributor
initial-stock-distributor
0
1000
98
1
1
NIL
HORIZONTAL

SLIDER
17
324
189
357
initial-stock-retailer
initial-stock-retailer
0
500
100
1
1
NIL
HORIZONTAL

SLIDER
20
103
192
136
num-of-distributors
num-of-distributors
1
10
2
1
1
NIL
HORIZONTAL

SLIDER
20
144
192
177
num-of-retailers
num-of-retailers
1
10
1
1
1
NIL
HORIZONTAL

MONITOR
761
16
869
69
Demand today
demand-today
0
1
13

TEXTBOX
566
83
744
101
Type of distribution for demand
10
0.0
1

TEXTBOX
568
160
770
178
parameters for deterministic demand
10
0.0
1

TEXTBOX
568
224
742
242
parameters for poisson distribution
10
0.0
1

TEXTBOX
566
296
799
314
parameters for truncated normal distribution
10
0.0
1

PLOT
760
76
1025
266
on-hand inventory
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS

PLOT
762
276
1026
457
total profit
NIL
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS

SLIDER
18
370
190
403
record-length
record-length
1
10
10
1
1
NIL
HORIZONTAL

SLIDER
571
16
743
49
days-of-simulation
days-of-simulation
500
10000
500
100
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

Inventory management is a critical problem in the supply chain management. Lots of analysis regarding the inventory dynamics in the supply chain system can be found in the literature of the operations research community. In this model, we explore a simplified network supply chain. There is one supplier, which has unlimited supply. There can be multiple distributors and retailers. Players order from upstream nodes and ship to downstream nodes. By selling goods, the players receive revenues. If players are not able to satisfy the demand, back-orders are created, the penalty costs occur. Also holding inventory incurs utility and maitenance costs. Players are striving to maximize their profit by wisely manage their ordering strategy and inventory policy.

## HOW IT WORKS

Every day, a daily demand is imposed to each retailer. Retailers order from distributors, and distributors order from the supplier. We assume that all distributors and retailers are following the base-stock policy, that is: once the inventory position drops below the base-stock level, the player order the difference between the base-stock level and current inventory position; otherwise order nothing. The retailers will choose which distributor to order from according to the back-orders. The distributors need to allocate the available on-hand inventory to retailers, according to their respective quota. Once all the order, replenish, and ship activities are complished, we conclude the day and summarize the cost and revenue for each player. Then a new day starts.

## HOW TO USE IT

Press "setup" button to establish the supply chain. Press "go" button to carry on the simulation.

"number-of-distributors" and "number-of-retailers" sliders adjust the number of distributors and retailers in the supply chain system.

"lead-time-supplier-distributor" indicates the number of days of supply delay between the supplier and distributors. Similarly, "lead-time-distributor-retailer" indicates the number of days of supply delay between distributors and retailers.

"initial-stock-distributor" and "initial-stock-retailer" are the initial inventory at the begining of the simulation for distributors and retailers, respectively.

"record-length" indicates how many recent day's demand history will be kept in record.

If ticks > "days-of-simulation", the program stops.

There are a chooser and several sliders for adjusting the generation mechanism of the daily demand. The monitor indicates today's demand.

The "total profit" and "on-hand inventory" plot the profit and on-hand stock level, respectively, at the current day of simulation for all players.

## THINGS TO NOTICE

The fluctuation of the on-hand inventory level at the distributors and retailers. It is said the fluctuation would propogate from downstream to upstream, so-called bull-whip effect.

## THINGS TO TRY

Users can adjust the lead times to see how this will affect the supply chain. 

Also, will the safety factors have significant influence on the supply chain?

How will different demand distributions matter?

## EXTENDING THE MODEL

In the current model, we assume that all the lead times are the same. Actually they can be different. This would cause desyncronization, which may lead to interesting results.

In the current model, we assume full connection. However, a retailer may not have connections with all the distributors. Will this create bias?

In the current model, we only consider 3 echelons. It may be interesting to extend to more echelons and more complicated supply chain configuration.

## RELATED MODELS

Wilensky, U. and Stroup, W. (2003). NetLogo HubNet Root Beer Game model. http://ccl.northwestern.edu/netlogo/models/HubNetRootBeerGame. Center for Connected Learning and Computer-Based Modeling, Northwestern Institute on Complex Systems, Northwestern University, Evanston, IL.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

arc
3.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Polygon -7500403 true true 150 150 105 225 195 225 150 150

@#$#@#$#@
0
@#$#@#$#@
