;Dr. Lin Xiang created this module at the University of Kentucky in 2022. If you mention this model in a publication, we ask that you include the citations below.
;
;Xiang, L. (2022). Ocean Acidification. Department of STEM Education, University of Kentucky, Lexington, KY.
;
;This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 4.0 License. To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/4.0/.
;
;References:
;Millero, F. J.; Graham, T. B.; Huang, F.; Bustos-Serrano, H.; Pierrot, D. Dissociation constants of carbonic acid as a function of salinity and temperature. (2006). Marine Chemistry, 100 (1-2), 80-94.
;
;This model is inspired by the KCVS applets from the 2100 C.E. RCP projections. https://applets.kcvs.ca/OceanAcidification/oceanAcid.html?rcp=2.6
;;------------------------------


breed [factories factory]
breed [CO2s CO2]
breed [H2CO3s H2CO3]
breed [HCO3s HCO3]
breed [CO3s CO3]
breed [H3OS H30]
breed [OHs OH]
breed [legends legend]

turtles-own[ ]
globals [
  Num-H
  Num-OH
  Num-CO2
  Num-H2CO3
  Num-HCO3
  Num-CO3
  current-ph
  current-pOH
  current-conc-H
  current-conc-OH
  current-conc-H2CO3s
  current-conc-HCO3s
  current-conc-CO3s
  M-H2CO3
  M-HCO3
  M-CO3
  M-H
  M-OH
  K-co2
  Ka1
  Ka2
  change-1
  change-2
  reaction-num
  ]

to setup
  ca

  set-bk
  set-legends
  set-shapes
  set-tt
  display-particles
  hiding-ions
  output


  reset-ticks

end

to set-bk
  ask patches with [pycor > 35 ] [set pcolor 86]
  ask patches with [pycor <= 35 and pycor >= 30] [set pcolor (-0.5 * pycor + 104)]
  ask patches with [pycor < 30 ] [set pcolor 96.5]
  ask patches with [ pycor = max-pycor or pycor < 6] [set pcolor 101]
  ask patches with [ pxcor = min-pxcor or pxcor = max-pxcor] [set pcolor 101]

end


to set-tt

  set K-co2 3.886647 * 10 ^ (-8)   ;calculater based on T CO2 0.002 and 250 ppm
  set Ka1 10 ^ (-5.903)         ;from Millero et al. (2006)
  set Ka2 10 ^ (-9.702)         ;;from Millero et al. (2006)

  ;set CO2 in air
  set p-co2 250                  ;when initiating model, PCO2 is always 250. This helps fix students' beginning exploration. It could be less convenient during exploration.
  set current-conc-H 6.29624889 * 10 ^ (-9)
  set current-ph (- log current-conc-H 10) ; 8.201          ;set the initital ph based on p-CO2 referenced from CO2sys 2.1
  set num-CO2 round (P-CO2 / 10)
  create-CO2s num-CO2  [set-tt-properties-air]

  ;set initial carbon species
  set current-conc-H2CO3s 0.0000097166175      ; based on Total C 0.002
  set current-conc-HCO3s 0.00192942933173    ; based on Total 0.002
  set current-conc-CO3s 0.00006086210779    ; based on Total 0.002

  ;reach equlibria
  CO2-dissolve
  Ka1-equilibrium
  Ka2-equilibrium

  ;set visual numbers
  set num-H2CO3 round (5000 * 3 ^ (log current-conc-H2CO3s 10))
  set num-HCO3 round (5000 * 3 ^ (log current-conc-HCO3s 10))
  set num-CO3 round (5000 * 3 ^ (log current-conc-CO3s 10))

  create-H2CO3s Num-H2CO3  [set-tt-properties-aq]
  create-HCO3s Num-HCO3  [set-tt-properties-aq]
  create-CO3s Num-CO3  [set-tt-properties-aq]

  set num-H round (500 * 1.3895 ^ (-1 * current-ph))
  set num-OH round (500 * 1.3895 ^ (-1 * (14 - current-ph)))

  create-h3os num-H [set-tt-properties-aq ]
  create-ohs num-OH [set-tt-properties-aq ]

  ;set factory
  create-factories 1
  [setxy 12 32 set color 35 set size 6 ]
 end



to set-tt-properties-aq
  setxy 1 + random-float (max-pxcor - 2) 6 + random-float (max-pycor - 15)
  (ifelse
    choose-a-visual = "Molecule visual" [set size 1.25]
    choose-a-visual = "Symbolic visual" [set size 1.75])
end

to set-tt-properties-air
  setxy 1 + random-float (max-pxcor - 2) (30 + random-float 7)
   (ifelse
    choose-a-visual = "Molecule visual" [set size 1.25]
    choose-a-visual = "Symbolic visual" [set size 1.75])
end


to set-legends
 create-legends 12 [set label-color white]

 ask legend 0[setxy 4 3.5]

 ask legend 1
   [set shape "blank" set size 1.5
    set label "Hydroxide(OH-)" setxy 13.3 3.9]

  ask legend 2 [setxy 4 1]

  ask legend 3
  [set shape "blank" set size 1.5
    set label "Hydronium (H3O+)" setxy 14.5 1.3]

  ask legend 4[setxy 20 3.5]

  ask legend 5
  [set shape "blank" set size 1.5
    set label "Carbon dioxide (CO2)" setxy 32.4 3.9]

  ask legend 6 [setxy 39 3.5]

  ask legend 7
  [set shape "blank" set size 1.5
    set label "Carbonic acid (H2CO3)" setxy 52 3.9]

   ask legend 8 [setxy 20 1]

  ask legend 9
  [set shape "blank" set size 1.5
    set label "Bicarbonate (HCO3-)" setxy 32 1.3]

   ask legend 10[setxy 39 1]

  ask legend 11
  [set shape "blank" set size 1.5
    set label "Carbonate (CO3(2-))" setxy 50.5 1.3]

end


to go

set-shapes-1
aq-move
H2CO3-move
CO2-move
repeat 1000 [
    CO2-dissolve
    Ka1-equilibrium
    Ka2-equilibrium
    set reaction-num reaction-num + 1]
display-particles
hiding-ions
output


tick

end

to go-1

set-shapes-1
aq-move
H2CO3-move
CO2-move
    CO2-dissolve tick
    Ka1-equilibrium tick
    Ka2-equilibrium
display-particles
hiding-ions
output


tick

end

to hiding-ions
  if Hide-CO2-&-Carbonic-acid? = true
    [if any? H2CO3s [ask H2CO3s [die]]
     if any? CO2s [ask CO2s [die]]]

  if Hide-Bicarbonate-ions? = true
    [if any? HCO3s [ask HCO3s [die]]]

  if Hide-Carbonate-ions? = true
    [if any? CO3s [ask CO3s [die]]]

  if Hide-H-&-OH-ions? = true
    [if any? H3Os [ask H3Os [die]]
     if any? OHs [ask OHs [die]]]
end

to display-particles
  set num-CO2 round (P-CO2 / 10)
  set num-H2CO3 round (5000 * 3 ^ (log current-conc-H2CO3s 10))
  set num-HCO3 round (5000 * 3 ^ (log current-conc-HCO3s 10))
  set num-CO3 round (5000 * 3 ^ (log current-conc-CO3s 10))


ifelse hide-CO2-&-Carbonic-acid? = false
  [
   let current-CO2 count CO2s
   let current-H2CO3 count H2CO3s

    (ifelse
    current-CO2 > num-CO2
     [ask up-to-n-of (current-CO2 - num-CO2) CO2s [die]]
    current-CO2 < num-CO2
     [create-CO2s (num-CO2 - current-CO2)
        [ifelse random 2 = 0 [setxy 10 33 set size 1.25 set heading -90 + random 180]
          [set-tt-properties-air]
        ]])

  (ifelse
    current-H2CO3 > num-H2CO3
     [ask up-to-n-of (current-H2CO3 - num-H2CO3) H2CO3s [die]]
    current-H2CO3 < num-H2CO3
     [create-H2CO3s (num-H2CO3 - current-H2CO3) [set-tt-properties-aq]])
    ]

   [if any? H2CO3s [ask H2CO3s [die]]
    if any? CO2s [ask CO2s [die]]]

ifelse Hide-Bicarbonate-ions? = false
    [let current-HCO3 count HCO3s
  (ifelse
    current-HCO3 > num-HCO3
     [ask up-to-n-of (current-HCO3 - num-HCO3) HCO3s [die]]
    current-HCO3 < num-HCO3
     [create-HCO3s (num-HCO3 - current-HCO3) [set-tt-properties-aq]])]
    [if any? HCO3s [ask HCO3s [die]]]

ifelse Hide-Carbonate-ions? = false
    [let current-CO3 count CO3s
  (ifelse
    current-CO3 > num-CO3
     [ask up-to-n-of (current-CO3 - num-CO3) CO3s [die]]
    current-CO3 < num-CO3
     [create-CO3s (num-CO3 - current-CO3) [ set-tt-properties-aq]])]
    [if any? CO3s [ask CO3s [die]]]

ifelse Hide-H-&-OH-ions? = false
    [ H-OH-equilibrium]
    [if any? H3Os [ask H3Os [die]]
     if any? OHs [ask OHs [die]]]
end

to CO2-dissolve
  set current-conc-H2CO3s (P-CO2 * K-co2)

end

to Ka1-equilibrium

  set current-conc-H 10 ^ (- current-ph)

  let Q1 current-conc-H * current-conc-HCO3s / current-conc-H2CO3s

  (ifelse
   (Q1 - Ka1) < Ka1 * -0.00001          ; if Q1 < Ka1 , reaction forward
  [set change-1 ((-1 * (current-conc-HCO3s + current-conc-H) + sqrt ((current-conc-HCO3s + current-conc-H) ^ 2 + 4 * (Ka1 * current-conc-H2CO3s - current-conc-HCO3s * current-conc-H))) / 2)
   set current-conc-H current-conc-H + change-1
   set current-conc-HCO3s current-conc-HCO3s + change-1
   set current-conc-H2CO3s current-conc-H2CO3s - change-1
   set current-ph (- log current-conc-H 10)
  ]

   (Q1 - Ka1) > Ka1 * 0.00001          ; if Q1 > Ka1 , reaction reverse
  [set change-1 ((current-conc-HCO3s + current-conc-H + Ka1) - sqrt ((current-conc-HCO3s + current-conc-H + Ka1) ^ 2 + 4 * (Ka1 * current-conc-H2CO3s - current-conc-HCO3s * current-conc-H))) / 2
   set current-conc-H current-conc-H - change-1
   set current-conc-HCO3s current-conc-HCO3s - change-1
   set current-conc-H2CO3s current-conc-H2CO3s + change-1
   set current-ph (- log current-conc-H 10)
  ])


end

to Ka2-equilibrium

 set current-conc-H 10 ^ (- current-ph)

 let Q2 current-conc-H * current-conc-CO3s / current-conc-HCO3s

 (ifelse
    (Q2 - Ka2) < Ka2 * -0.00001            ;if Q2 < Ka2 , reaction forward
  [set change-2 (-0.5 * ((current-conc-H + current-conc-CO3s + Ka2) - sqrt ((current-conc-H + current-conc-CO3s + Ka2) ^ 2 + 4 * (Ka2 * current-conc-HCO3s - current-conc-H * current-conc-CO3s))))
   set current-conc-H current-conc-H + change-2
   set current-conc-CO3s current-conc-CO3s + change-2
   set current-ph (- log current-conc-H 10)
   set current-conc-HCO3s current-conc-HCO3s - change-2]

     (Q2 - Ka2) > Ka2 * 0.00001            ;if Q2 > Ka2 , reaction reverse
  [set change-2 ((current-conc-H + current-conc-CO3s + Ka2) - sqrt ((current-conc-H + Ka2 + current-conc-CO3s) ^ 2 + 4 * (Ka2 * current-conc-HCO3s - current-conc-H * current-conc-CO3s))) / 2
   set current-conc-H current-conc-H - change-2
   set current-conc-CO3s current-conc-CO3s - change-2
   set current-ph (- log current-conc-H 10)
   set current-conc-HCO3s current-conc-HCO3s + change-2])

end

to H-OH-equilibrium

  set num-H round (500 * 1.3895 ^ (-1 * current-ph))
  set num-OH round (500 * 1.3895 ^ (-1 * (14 - current-ph)))

  let current-H count H3Os
  let current-OH count OHs

  (ifelse
    num-H > current-H
     [create-h3os abs (num-H - current-H) [set-tt-properties-aq ]]
    num-H < current-H
    [ask n-of abs (num-H - current-H) h3os [die]])

   (ifelse
    num-OH > current-OH
    [create-ohs abs (num-OH - current-OH) [set-tt-properties-aq ]]
    num-OH < current-OH
    [ask n-of abs (num-OH - current-OH) OHs [die]])

end


to CO2-move
  ask CO2s
  [(ifelse
    [pcolor] of patch-at dx dy = 101 [rt random 360]    ;if it's boundary ahead, rotate
    [pcolor] of patch-at dx dy <= 89 and [pcolor] of patch-at dx dy >= 86  [fd 0.2]; rt random 360] ;if the patch-ahead has the proper color,move forward
    [pcolor] of patch-at dx dy = 96.5                     ;if water ahead,
       [ifelse random 100 < 20 [set heading 180 fd 1 set breed H2CO3s] [rt random 360]])]  ;dissolve at a certain rate, create the animation
end

to H2CO3-move
   ask H2CO3s
    [if [pcolor] of patch-at dx dy = 89                ;if air aheas,
      [ifelse random 100 < 20
        [set heading 0 fd 1 set breed CO2s]
        [rt random 360]
    ]]
end

to aq-move
  ask turtles with [breed != CO2s and breed != legends]    ;ask aq turtles
     [(ifelse
        [pcolor] of patch-at dx dy = 101 or [pcolor] of patch-at dx dy = 89  [rt random 360]    ;if it's boundary ahead, rotate
        [pcolor] of patch-at dx dy = 96.5  [fd 0.2 rt random 360 if ycor < 6 [set ycor 6]] ;if the patch-ahead has the proper color, move forward.If ycor is low, put ycor higher
          )]
end



to set-shapes
  (ifelse
    choose-a-visual = "Molecule visual"
    [set-default-shape ohs "ab-oh-1"
     set-default-shape H3Os "ab-h3o-1"
     set-default-shape CO2s "ab-CO2"
     set-default-shape H2CO3s "ab-H2CO3"
     set-default-shape HCO3s "ab-HCO3"
     set-default-shape CO3s "ab-CO3"
     set-default-shape factories "factory"
      ask legend 0 [set shape "ab-oh-1" set size 1.75]
      ask legend 2 [set shape "ab-h3o-1" set size 1.75]
      ask legend 4 [set shape "ab-CO2" set size 1.75]
      ask legend 6 [set shape "ab-H2CO3" set size 1.75]
      ask legend 8 [set shape "ab-HCO3" set size 1.75]
      ask legend 10 [set shape "ab-CO3" set size 1.75]

    ]

   choose-a-visual = "Symbolic visual"
    [set-default-shape ohs "ab-oh-2"
     set-default-shape H3Os "ab-h3o-2"
     set-default-shape CO2s "ab-CO2-1"
     set-default-shape H2CO3s "ab-H2CO3-1"
     set-default-shape HCO3s "ab-HCO3-1"
     set-default-shape CO3s "ab-CO3-1"
     set-default-shape factories "factory"
      ask legend 0 [set shape "ab-oh-2" set size 1.75]
      ask legend 2 [set shape "ab-h3o-2" set size 1.75]
      ask legend 4 [set shape "ab-CO2-1" set size 1.75]
      ask legend 6 [set shape "ab-H2CO3-1" set size 1.75]
      ask legend 8 [set shape "ab-HCO3-1" set size 1.75]
      ask legend 10 [set shape "ab-CO3-1" set size 1.75]
  ])
end

to set-shapes-1
  (ifelse
    choose-a-visual = "Molecule visual"
    [set-default-shape ohs "ab-oh-1"
     set-default-shape H3Os "ab-h3o-1"
     set-default-shape CO2s "ab-CO2"
     set-default-shape H2CO3s "ab-H2CO3"
     set-default-shape HCO3s "ab-HCO3"
     set-default-shape CO3s "ab-CO3"
     set-default-shape factories "factory"
     ask ohs [set shape "ab-oh-1" set size 1.25]
     ask H3Os [set shape "ab-h3o-1" set size 1.25]
     ask CO2s [set shape "ab-CO2" set size 1.25]
     ask H2CO3s [set shape "ab-H2CO3" set size 1.25]
     ask HCO3s [set shape "ab-HCO3" set size 1.25]
     ask CO3s [set shape "ab-CO3" set size 1.25]
      ask legend 0 [set shape "ab-oh-1" set size 1.75]
      ask legend 2 [set shape "ab-h3o-1" set size 1.75]
      ask legend 4 [set shape "ab-CO2" set size 1.75]
      ask legend 6 [set shape "ab-H2CO3" set size 1.75]
      ask legend 8 [set shape "ab-HCO3" set size 1.75]
      ask legend 10 [set shape "ab-CO3" set size 1.75]

    ]

   choose-a-visual = "Symbolic visual"
    [set-default-shape ohs "ab-oh-2"
     set-default-shape H3Os "ab-h3o-2"
     set-default-shape CO2s "ab-CO2-1"
     set-default-shape H2CO3s "ab-H2CO3-1"
     set-default-shape HCO3s "ab-HCO3-1"
     set-default-shape CO3s "ab-CO3-1"
     set-default-shape factories "factory"
     ask ohs [set shape "ab-oh-2" set size 1.75]
     ask H3Os [set shape "ab-h3o-2" set size 1.75]
     ask CO2s [set shape "ab-CO2-1" set size 1.75]
     ask H2CO3s [set shape "ab-H2CO3-1" set size 1.75]
     ask HCO3s [set shape "ab-HCO3-1" set size 1.75]
     ask CO3s [set shape "ab-CO3-1" set size 1.75]
      ask legend 0 [set shape "ab-oh-2" set size 1.75]
      ask legend 2 [set shape "ab-h3o-2" set size 1.75]
      ask legend 4 [set shape "ab-CO2-1" set size 1.75]
      ask legend 6 [set shape "ab-H2CO3-1" set size 1.75]
      ask legend 8 [set shape "ab-HCO3-1" set size 1.75]
      ask legend 10 [set shape "ab-CO3-1" set size 1.75]
  ])
end

to output
  output-type P-CO2
  repeat 3 [output-type " "]
  output-write precision (current-conc-H2CO3s * 1000000) 5
  repeat 3 [output-type " "]
  output-write precision (current-conc-HCO3s * 1000000) 5
  repeat 3 [output-type " "]
  output-type precision (current-conc-CO3s * 1000000) 5
  repeat 3 [output-type " "]
  output-type precision (current-conc-H) 12
  repeat 3 [output-type " "]
  output-type precision current-ph 3
  repeat 3 [output-type " "]
  output-type precision (current-conc-H * current-conc-HCO3s / current-conc-H2CO3s) 9
  repeat 3 [output-type " "]
  output-print precision (current-conc-H * current-conc-CO3s / current-conc-HCO3s) 13
  end
@#$#@#$#@
GRAPHICS-WINDOW
256
10
795
370
-1
-1
9.0
1
10
1
1
1
0
1
1
1
0
58
0
38
1
1
1
ticks
10.0

BUTTON
147
31
249
76
Set up/Restart
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
6
103
248
136
Run 3 coupled reactions continuously
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

PLOT
799
316
1103
637
Aqueous carbon concentrations
Time
Log concentration
0.0
10.0
-6.0
-2.0
true
true
"" ""
PENS
"Carbonic acid" 1.0 0 -4079321 true "" "plot (log current-conc-H2CO3s 10)"
"Bicarbonate" 1.0 0 -8630108 true "" "plot (log current-conc-HCO3s 10)"
"Carbonate" 1.0 0 -10899396 true "" "plot (log current-conc-CO3s 10)"

MONITOR
254
600
341
637
H2CO3 (umol/L)
current-conc-H2CO3s * 1000000
5
1
9

SLIDER
6
376
247
409
P-CO2
P-CO2
250
900
250.0
5
1
ppm
HORIZONTAL

MONITOR
607
600
657
637
pH
current-ph
3
1
9

BUTTON
8
271
248
304
(H2CO3) <--> (H+) + (HCO3-)
Ka1-equilibrium\ndisplay-particles\noutput\n\ntick
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
8
305
248
339
(HCO3-) <--> (H+) + (CO3(2-))
Ka2-equilibrium\ndisplay-particles\noutput\n\ntick
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
341
600
432
637
HCO3- (umol/L)
current-conc-HCO3s * 1000000
5
1
9

MONITOR
432
600
526
637
CO3(2-) (umol/L)
current-conc-CO3s * 1000000
5
1
9

BUTTON
8
237
248
270
(CO2) + (H2O) <--> (H2CO3)
CO2-dissolve\ndisplay-particles\noutput\n\ntick
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

MONITOR
526
600
607
637
H3O+ (mol/L)
current-conc-H
14
1
9

MONITOR
722
600
796
637
Q2
current-conc-H * current-conc-CO3s / current-conc-HCO3s
13
1
9

MONITOR
657
600
722
637
Q1
current-conc-H * current-conc-HCO3s / current-conc-H2CO3s
9
1
9

BUTTON
6
138
248
171
Run 3 coupled reactions 1000 times
set-shapes-1\naq-move\nH2CO3-move\nCO2-move\nrepeat 1000 [\n    CO2-dissolve\n    Ka1-equilibrium\n    Ka2-equilibrium\n    set reaction-num reaction-num + 1]\ndisplay-particles\noutput\n\n\ntick
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

PLOT
799
10
1102
312
pH
Time
pH
0.0
10.0
7.5
8.5
true
false
"" ""
PENS
"default" 1.0 0 -2674135 true "" "plot current-ph"

TEXTBOX
661
640
807
658
Ka1:1.25E-6 | Ka2:1.986E-10                   
9
125.0
1

SWITCH
6
467
234
500
Hide-CO2-&-Carbonic-acid?
Hide-CO2-&-Carbonic-acid?
1
1
-1000

SWITCH
6
502
214
535
Hide-Bicarbonate-ions?
Hide-Bicarbonate-ions?
1
1
-1000

SWITCH
6
537
201
570
Hide-Carbonate-ions?
Hide-Carbonate-ions?
1
1
-1000

SWITCH
6
572
182
605
Hide-H-&-OH-ions?
Hide-H-&-OH-ions?
1
1
-1000

TEXTBOX
6
10
239
28
1: Choose a visual style and set up model.
10
125.0
1

TEXTBOX
5
86
192
104
2: Couple three reactions
10
125.0
1

TEXTBOX
6
431
270
457
Optional:  Show/hide some ions to enhance observations
10
125.0
1

CHOOSER
6
31
139
76
Choose-a-visual
Choose-a-visual
"Molecule visual" "Symbolic visual"
1

TEXTBOX
5
356
200
374
3: Change pCO2 and observe results.\n
10
125.0
1

TEXTBOX
2
218
152
236
Or, run them separatedly
10
125.0
1

TEXTBOX
261
383
763
409
           | Carbon Species Concentration (umol/L)  |       Ocean Acidity         |\npCO2  |  H2CO3           HCO3-              CO3(2-)   |   H3O+ (mol/L)      pH   |        Q1                   Q2
10
125.0
1

BUTTON
6
173
248
206
Run 3 coupled reactions once
go-1
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

OUTPUT
254
412
796
597
10

TEXTBOX
6
615
221
637
Version info:--------------------\nThis version was updated on March 25, 2024.
9
125.0
1

@#$#@#$#@
## WHAT IS IT?

Today's average ocean pH has dropped from 8.2 to about 8.1 as the ocean has taken up a large amount of CO2 from human activities since the industrial revolution. One aspect of ocean acidification that has confused many students is the decrease in carbonate ion concentration in the ocean resulting from the increase in oceanic CO2 uptake. This model provides students with an established oceanic carbon system. Students may uncouple the three reactions involving the changes in three inorganic carbon species as well as explore the long-term system dynamics resulting from the coupled reactions.  

## HOW IT WORKS

Six particles are presented in the model: Carbon dioxide, Carbonic acid, bicarbonate ion, carbonate ion, hydronium ion, and hydroxide ion. Two visual styles are provided in the model. The icon visual is more user-friendly and the molecule visual provides more authentic molecule properties. The numbers of particles are transformed from the corresponding ion concentrations in the model. Since the H3O+ concentration is much lower than those of carbon species, a different equation is used to increase the number of H3O+ and OH- so that their changes are more visible.   
  

In the model, a total DIC concentration of 0.002 (mol kg-1) is used initially. The values of two equilibrium constants, Ka1 = 10 ^ (-5.903) and Ka2 = 10 ^ (-9.702), obtained from Millero et al. (2006), are used to calculate the concentrations of three species in a closed system where salinity is 3.3758% and temperature is 20 degree Celsius. The concentrations of Carbonic acid (H2CO3), bicarbonate ion (HCO3-), carbonate ion (CO3(2-)) are calculated using the following equations:

1)  [H2CO3] = K-co2 * P-co2 (partial pressures of carbon dioxide)
2)  [HCO3-][H+]/[H2CO3]=Ka1
3)  [CO3--][H+]/[HCO3]=Ka2

The initial ocean pHs and partial pressures of carbon dioxide were obtained from CO2SYS (https://cdiac.ess-dive.lbl.gov/ftp/co2sys/). The resulting pH in the model is calibrated based on CO2SYS and the average pH deviation is 0.5%. 


## HOW TO USE IT

1. Choose a visual style, and then click on the **set up/restart** button the confirm the setting.

2. Use the **Run 3 coupled reactions continuously** to run the three reactions in sequence, i.e.,[CO2+H2O<-->H2CO3; H2CO3<-->(HCO3-)+(H+);(HCO3-)<-->(CO3--)+(H+)]   repeatedly. While the model is running, you may change the **P-CO2** slider, representing the partial pressures of carbon dioxide, to view the corresponding changes in oceanic pH and the concentrations of three carbon species. 

3. The **Run 3 coupled reactions 1000 times** button and **Run 3 coupled reactions once** button allow you to run the three reactions in sequence 1000 times or once per click.

4. To observe the dynamic changes in depth, you may also run the three reactions individually using the buttons 
**1-(CO2) + (H2O) <--> (H2CO3)**, 
**2-(H2CO3) <--> (H+) + (HCO3-)**, and
 **3-(HCO3-) <--> (H+) + (CO3(2-))** 


## THINGS TO TRY AND NOTICE

* Run the system using the **Run 3 coupled reactions continuously** button. Drag the **P-CO2** slider to increase or decrease the partial pressures of carbon dioxide in the air. Observe 1) how the concentrations of three carbon species changes accordingly and 2) how oceanic pH value change accordingly. 

* The changes in HCO3- might not be very visible in plots. Use the numeric monitors to track the changes in ion concentrations.

* Use switches to show/hide some ions to focus your observation. For example, only show H2CO3 and hide the rest ions. 

* Apply equilibrium constants and reaction quotients to explain the concentration changes in the model.  

## RELATED MODELS

Find the **learning activity worksheets** and more chemistry models at https://sites.google.com/view/3d-science-abm/physics-chemistry   

## CREDITS AND REFERENCES

Dr. Lin Xiang created this module at the University of Kentucky in 2022. If you mention this model in a publication, we ask that you include the citations below.

Xiang, L. (2022). Ocean Acidification. Department of STEM Education, University of Kentucky, Lexington, KY.

![CC BY-NC-SA 4.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 4.0 License. To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/4.0/.

References:
Millero, F. J.; Graham, T. B.; Huang, F.; Bustos-Serrano, H.; Pierrot, D. Dissociation constants of carbonic acid as a function of salinity and temperature. (2006). Marine Chemistry, 100 (1-2), 80-94.

This model is inspired by the KCVS applets from the 2100 C.E. RCP projections. https://applets.kcvs.ca/OceanAcidification/oceanAcid.html?rcp=2.6
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

ab-co2
true
0
Circle -2674135 true false 180 90 120
Circle -16777216 true false 75 75 150
Circle -2674135 true false 0 90 120

ab-co2-1
true
0
Circle -2064490 true false 90 90 120

ab-co3
true
0
Circle -2674135 true false 120 0 120
Circle -16777216 true false 60 60 180
Circle -2674135 true false 0 90 120
Circle -2674135 true false 150 165 120

ab-co3-1
true
0
Polygon -1 true false 150 30 30 225 270 225

ab-h2co3
true
0
Circle -2674135 true false 120 0 120
Circle -16777216 true false 60 60 180
Circle -2674135 true false 0 90 120
Circle -2674135 true false 150 165 120
Circle -1 true false 122 0 42
Circle -1 true false 163 253 42

ab-h2co3-1
true
0
Polygon -1184463 true false 150 30 30 225 270 225
Circle -955883 true false 116 11 67
Circle -955883 true false 11 176 67

ab-h2o-1
true
0
Circle -7500403 false true 75 75 148
Circle -7500403 true true 84 189 42
Circle -7500403 true true 204 144 42

ab-h3o-1
true
0
Circle -1 true false 114 54 42
Circle -2674135 true false 75 75 150
Circle -1 true false 69 189 42
Circle -1 true false 204 159 42

ab-h3o-2
true
0
Polygon -2674135 true false 105 75 195 75 240 150 195 225 105 225 60 150

ab-hco3
true
0
Circle -2674135 true false 120 0 120
Circle -16777216 true false 60 60 180
Circle -2674135 true false 0 90 120
Circle -2674135 true false 150 165 120
Circle -1 true false 163 253 42

ab-hco3-1
true
0
Polygon -8630108 true false 150 30 30 225 270 225
Circle -955883 true false 116 11 67

ab-oh-1
true
0
Circle -13345367 true false 75 75 150
Circle -1 true false 189 174 42

ab-oh-2
true
0
Polygon -13345367 true false 105 75 195 75 240 150 195 225 105 225 60 150

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

blank
true
0

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

factory
false
0
Rectangle -7500403 true true 76 194 285 270
Rectangle -7500403 true true 36 95 59 231
Rectangle -16777216 true false 90 210 270 240
Line -7500403 true 90 195 90 255
Line -7500403 true 120 195 120 255
Line -7500403 true 150 195 150 240
Line -7500403 true 180 195 180 255
Line -7500403 true 210 210 210 240
Line -7500403 true 240 210 240 240
Line -7500403 true 90 225 270 225
Circle -1 true false 37 73 32
Circle -1 true false 55 38 54
Circle -1 true false 96 21 42
Circle -1 true false 105 40 32
Circle -1 true false 129 19 42
Rectangle -7500403 true true 14 228 78 270

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.3.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
